defmodule Console.Ecran.Plateau do
  @moduledoc """
  Écran appelé pour afficher un jeu de cartes en cours.
  """

  @behaviour Console.Ecran

  defstruct joueur: :inconnu, partie: :inconnue

  @typedoc "Un joueur"
  @type joueur() :: %{id: integer(), nom: String.t()}

  @typedoc "Une partie"
  @type partie() :: %{id: integer(), processus: pid(), titre: String.t()}

  @typedoc "Un écran `Console.Ecran.Plateau`"
  @type t() :: %{joueur: joueur(), partie: partie()}

  @doc """
  Crée un écran.
  """
  @spec new(joueur(), partie()) :: t()
  def new(joueur = %{id: _, nom: _}, partie = %{id: _, processus: _, titre: _}) do
    %Console.Ecran.Plateau{joueur: joueur, partie: partie}
  end

  @impl true
  @doc """
  Retourne le titre de l'écran.
  """
  @spec titre(struct()) :: String.t()
  def titre(écran), do: écran.partie.titre

  @impl true
  @doc """
  Retourne le texte de l'écran.
  """
  @spec texte(struct()) :: String.t()
  def texte(écran) do
    coups_valides = coups_valides(écran)
    titres =
      Enum.map(coups(écran), fn {coup, affichage, _} ->
        index = Enum.find_index(coups_valides, fn {valeur, _, _} -> coup == valeur end)

        if index do
          "[#{index + 1}] - #{affichage}"
        else
          "[-] - #{affichage}"
        end
      end)

    """
    Entrez le numéro d'une carte ou d'un coup pour le jouer dans cette partie.
    Les coups précédés de [-] ne sont pas actuellement valides.

    #{Enum.join(titres, "\n")}
    """
  end

  @impl true
  @doc """
  Retourne le prompt de l'écran (le texte tout en bas, indiquant quelle information est à préciser ici).
  """
  @spec prompt(struct()) :: String.t()
  def prompt(_écran), do: "Entrez le numéro du coup à jouer : "

  @impl true
  @doc """
  Gère les entrées claviers de l'utilisateur.

  Cette fonction peut retourner plusieurs informations :

  - `:silence` : indique à l'interface qu'il n'est pas nécessaire d'afficher l'écran de novueau ;
  - `:prompt` : indique à l'interface qu'il est simplement nécessaire d'afficher de nouveau le prompt de l'écran, sans son texte ;
  - `:rafraîchir` : indique à l'interface qu'il faut afficher tout l'écran (titre, texte et prompt) ;
  - `{atome, texte}` : où `atome` est l'un des trois atomes ci-dessus et `texte` est le texte à afficher ;
  - `écran` : où 'écran' est la nouvelle structure du module `écran`.

  """
  @spec gérer_entrée(struct(), String.t()) :: Console.Ecran.retour_clavier()
  def gérer_entrée(écran, entrée) do
    with {index, _} <- Integer.parse(entrée),
         {:ok, coup} <- Enum.fetch(coups_valides(écran), index - 1) do
      {coup, affichage, _} = coup
      Jeu.jouer(écran.partie.id, écran.partie.processus, écran.joueur.id, coup)
      {:silence, affichage}
    else
      :error -> {:prompt, "Ce coup n'est pas valide."}
    end
  end

  defp coups(écran) do
    Partie.coups(écran.partie.processus, écran.joueur.id)
  end

  defp coups_valides(écran) do
    Enum.filter(coups(écran), fn {_, _, peut_jouer?} -> peut_jouer? end)
  end
end
